<?php
require_once 'config/db.php';

function getTasks() {
    try {
        $pdo = getDBConnection();
        $stmt = $pdo->query("SELECT * FROM tasks ORDER BY position ASC, created_at DESC");
        return $stmt->fetchAll();
    } catch (PDOException $e) {
        return [];
    }
}

function groupTasksByStatus($tasks) {
    $grouped = ['todo' => [], 'inprogress' => [], 'done' => []];
    foreach ($tasks as $task) {
        if (isset($grouped[$task['status']])) {
            $grouped[$task['status']][] = $task;
        }
    }
    return $grouped;
}

$tasks = getTasks();
$groupedTasks = groupTasksByStatus($tasks);
$priorityLabels = ['low' => 'Düşük', 'medium' => 'Orta', 'high' => 'Yüksek'];

$developer = [
    'name' => 'Burak Kılıç',
    'instagram' => 'klcburak.0',
    'website' => 'https://kodhane.com.tr'
];
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kanban Board</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600&display=swap" rel="stylesheet">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Inter', -apple-system, sans-serif;
            background: #f5f5f5;
            color: #1a1a1a;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 40px 24px;
            flex: 1;
        }
        
        .header {
            margin-bottom: 32px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 16px;
        }
        
        .header h1 { font-size: 22px; font-weight: 600; }
        
        .developer-info a {
            display: flex;
            align-items: center;
            gap: 6px;
            text-decoration: none;
            color: #666;
            font-size: 13px;
            padding: 8px 12px;
            background: #fff;
            border: 1px solid #e0e0e0;
            border-radius: 6px;
            transition: all 0.2s;
        }
        
        .developer-info a:hover { color: #111; border-color: #ccc; }
        .developer-info svg { width: 16px; height: 16px; }
        
        .add-form {
            background: #fff;
            border: 1px solid #e0e0e0;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 28px;
            display: flex;
            gap: 12px;
            flex-wrap: wrap;
            align-items: flex-end;
        }
        
        .form-group { display: flex; flex-direction: column; gap: 6px; }
        .form-group.grow { flex: 1; min-width: 180px; }
        .form-group label { font-size: 12px; font-weight: 500; color: #666; }
        
        .form-group input, .form-group select {
            padding: 10px 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 14px;
            font-family: inherit;
            background: #fff;
        }
        
        .form-group input:focus, .form-group select:focus { outline: none; border-color: #333; }
        .color-input { width: 40px; height: 40px; padding: 2px; border-radius: 6px; cursor: pointer; }
        
        .btn {
            padding: 10px 20px;
            background: #222;
            color: #fff;
            border: none;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s;
        }
        
        .btn:hover { background: #444; transform: translateY(-1px); }
        
        .board { display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px; }
        @media (max-width: 900px) { .board { grid-template-columns: 1fr; } }
        
        .column {
            background: #fff;
            border: 1px solid #e0e0e0;
            border-radius: 10px;
            min-height: 420px;
            display: flex;
            flex-direction: column;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .column.drag-over { 
            background: #f0f7ff; 
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
            transform: scale(1.01);
        }
        
        .column-header {
            padding: 16px 18px;
            border-bottom: 1px solid #eee;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }
        
        .column-title { display: flex; align-items: center; gap: 10px; }
        .column-dot { width: 8px; height: 8px; border-radius: 50%; transition: transform 0.3s; }
        .column.drag-over .column-dot { transform: scale(1.5); }
        .column-todo .column-dot { background: #eab308; }
        .column-inprogress .column-dot { background: #3b82f6; }
        .column-done .column-dot { background: #22c55e; }
        .column-title h2 { font-size: 14px; font-weight: 600; }
        .column-count { 
            font-size: 13px; 
            color: #999;
            background: #f5f5f5;
            padding: 2px 8px;
            border-radius: 10px;
            transition: all 0.3s;
        }
        .column.drag-over .column-count {
            background: #3b82f6;
            color: #fff;
        }
        
        .task-list { flex: 1; padding: 12px; display: flex; flex-direction: column; gap: 10px; }
        
        /* KART STİLLERİ */
        .task-card {
            background: #fafafa;
            border: 1px solid #eee;
            border-radius: 8px;
            padding: 14px;
            cursor: grab;
            border-left: 3px solid var(--task-color, #3b82f6);
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
        }
        
        .task-card:hover { 
            border-color: #ddd; 
            box-shadow: 0 4px 12px rgba(0,0,0,0.08);
            transform: translateY(-2px);
        }
        
        /* SÜRÜKLEME BAŞLANGICI */
        .task-card.dragging {
            opacity: 0.9;
            transform: rotate(3deg) scale(1.05);
            box-shadow: 0 20px 40px rgba(0,0,0,0.2);
            z-index: 1000;
            cursor: grabbing;
        }
        
        /* PLACEHOLDER - boşluk gösterici */
        .task-placeholder {
            background: linear-gradient(90deg, #e0e7ff 25%, transparent 25%, transparent 50%, #e0e7ff 50%, #e0e7ff 75%, transparent 75%, transparent);
            background-size: 20px 20px;
            border: 2px dashed #3b82f6;
            border-radius: 8px;
            min-height: 80px;
            animation: placeholder-move 0.5s linear infinite, placeholder-pulse 1s ease-in-out infinite;
        }
        
        @keyframes placeholder-move {
            0% { background-position: 0 0; }
            100% { background-position: 20px 0; }
        }
        
        @keyframes placeholder-pulse {
            0%, 100% { opacity: 0.5; }
            50% { opacity: 0.8; }
        }
        
        /* BIRAKMA ANI */
        .task-card.just-dropped {
            animation: drop-bounce 0.4s cubic-bezier(0.34, 1.56, 0.64, 1);
        }
        
        @keyframes drop-bounce {
            0% { transform: scale(1.1); opacity: 0.8; }
            50% { transform: scale(0.95); }
            100% { transform: scale(1); opacity: 1; }
        }
        
        /* TAMAMLANDI KOLONU */
        .column-done .task-card.just-dropped {
            animation: complete-celebration 0.6s ease;
        }
        
        @keyframes complete-celebration {
            0% { transform: scale(1.1); }
            25% { transform: scale(0.95) rotate(-2deg); }
            50% { transform: scale(1.05) rotate(2deg); }
            75% { transform: scale(0.98); }
            100% { transform: scale(1) rotate(0); }
        }
        
        /* YENİ GÖREV EKLENDİĞİNDE */
        .task-card.just-added {
            animation: slide-in-bounce 0.5s cubic-bezier(0.34, 1.56, 0.64, 1);
        }
        
        @keyframes slide-in-bounce {
            0% { opacity: 0; transform: translateY(-30px) scale(0.8); }
            60% { transform: translateY(5px) scale(1.02); }
            100% { opacity: 1; transform: translateY(0) scale(1); }
        }
        
        /* SİLİNİRKEN */
        .task-card.removing {
            animation: remove-shrink 0.3s ease forwards;
        }
        
        @keyframes remove-shrink {
            0% { opacity: 1; transform: scale(1); }
            100% { opacity: 0; transform: scale(0.8) translateX(20px); }
        }
        
        .task-top { display: flex; justify-content: space-between; align-items: flex-start; gap: 8px; margin-bottom: 6px; }
        .task-title { font-size: 14px; font-weight: 500; }
        .task-priority { font-size: 10px; font-weight: 600; padding: 2px 6px; border-radius: 3px; text-transform: uppercase; }
        .priority-low { background: #dcfce7; color: #15803d; }
        .priority-medium { background: #fef9c3; color: #a16207; }
        .priority-high { background: #fee2e2; color: #dc2626; }
        
        .task-desc { font-size: 13px; color: #666; margin-bottom: 10px; display: -webkit-box; -webkit-line-clamp: 2; -webkit-box-orient: vertical; overflow: hidden; }
        .task-footer { display: flex; justify-content: space-between; align-items: center; }
        .task-date { font-size: 12px; color: #999; }
        .task-actions { display: flex; gap: 4px; opacity: 0; transition: opacity 0.15s; }
        .task-card:hover .task-actions { opacity: 1; }
        
        .task-edit, .task-delete {
            background: none;
            border: none;
            color: #999;
            cursor: pointer;
            padding: 4px;
            border-radius: 4px;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.2s;
        }
        
        .task-edit:hover { color: #3b82f6; background: #eff6ff; transform: scale(1.1); }
        .task-delete:hover { color: #ef4444; background: #fee2e2; transform: scale(1.1); }
        
        .empty { flex: 1; display: flex; align-items: center; justify-content: center; color: #bbb; font-size: 14px; }
        
        .modal-overlay {
            position: fixed;
            top: 0; left: 0; right: 0; bottom: 0;
            background: rgba(0,0,0,0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 1000;
            animation: fade-in 0.2s ease;
        }
        
        @keyframes fade-in {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        .modal { 
            background: #fff; 
            border-radius: 12px; 
            width: 100%; 
            max-width: 420px; 
            margin: 20px;
            animation: modal-pop 0.3s cubic-bezier(0.34, 1.56, 0.64, 1);
        }
        
        @keyframes modal-pop {
            from { opacity: 0; transform: scale(0.9) translateY(-20px); }
            to { opacity: 1; transform: scale(1) translateY(0); }
        }
        
        .modal-header { display: flex; justify-content: space-between; align-items: center; padding: 16px 20px; border-bottom: 1px solid #eee; }
        .modal-header h3 { font-size: 16px; font-weight: 600; }
        .modal-close { background: none; border: none; font-size: 24px; color: #999; cursor: pointer; line-height: 1; transition: transform 0.2s; }
        .modal-close:hover { color: #333; transform: rotate(90deg); }
        .modal-form { padding: 20px; }
        .modal-field { margin-bottom: 16px; }
        .modal-field label { display: block; font-size: 12px; font-weight: 500; color: #666; margin-bottom: 6px; }
        .modal-field input, .modal-field select { width: 100%; padding: 10px 12px; border: 1px solid #ddd; border-radius: 6px; font-size: 14px; font-family: inherit; transition: border-color 0.2s, box-shadow 0.2s; }
        .modal-field input:focus, .modal-field select:focus { outline: none; border-color: #3b82f6; box-shadow: 0 0 0 3px rgba(59,130,246,0.1); }
        .modal-row { display: flex; gap: 12px; }
        .modal-row .modal-field { flex: 1; }
        .modal-actions { display: flex; gap: 10px; justify-content: flex-end; margin-top: 20px; }
        .btn-cancel { padding: 10px 20px; background: #f5f5f5; color: #333; border: 1px solid #ddd; border-radius: 6px; font-size: 14px; cursor: pointer; transition: all 0.2s; }
        .btn-cancel:hover { background: #eee; }
        
        footer {
            background: #fff;
            border-top: 1px solid #e0e0e0;
            padding: 20px 0;
            text-align: center;
        }
        
        .footer-content { max-width: 1400px; margin: 0 auto; padding: 0 24px; }
        
        .footer-content p {
            font-size: 13px;
            color: #999;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            flex-wrap: wrap;
        }
        
        .footer-content a {
            color: #666;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 4px;
            transition: color 0.2s;
        }
        
        .footer-content a:hover { color: #111; }
        .footer-content svg { width: 14px; height: 14px; }
        
        /* TOAST */
        .toast {
            position: fixed;
            bottom: 24px;
            right: 24px;
            background: #222;
            color: #fff;
            padding: 12px 20px;
            border-radius: 6px;
            font-size: 14px;
            z-index: 1001;
            animation: toast-in 0.3s cubic-bezier(0.34, 1.56, 0.64, 1);
        }
        
        @keyframes toast-in {
            from { opacity: 0; transform: translateY(20px) scale(0.9); }
            to { opacity: 1; transform: translateY(0) scale(1); }
        }
        
        .toast.error { background: #ef4444; }
        .toast.success { background: #22c55e; }
        
        /* KONFETİ - Tamamlandı kutlaması */
        .confetti {
            position: fixed;
            width: 10px;
            height: 10px;
            pointer-events: none;
            z-index: 9999;
            animation: confetti-fall 1s ease-out forwards;
        }
        
        @keyframes confetti-fall {
            0% { opacity: 1; transform: translateY(0) rotate(0deg); }
            100% { opacity: 0; transform: translateY(100px) rotate(720deg); }
        }
    </style>
</head>
<body>
    <div class="container">
        <header class="header">
            <h1>Kanban Board</h1>
            <div class="developer-info">
                <a href="https://instagram.com/<?= htmlspecialchars($developer['instagram']) ?>" target="_blank">
                    <svg viewBox="0 0 24 24" fill="currentColor">
                        <path d="M12 2.163c3.204 0 3.584.012 4.85.07 3.252.148 4.771 1.691 4.919 4.919.058 1.265.069 1.645.069 4.849 0 3.205-.012 3.584-.069 4.849-.149 3.225-1.664 4.771-4.919 4.919-1.266.058-1.644.07-4.85.07-3.204 0-3.584-.012-4.849-.07-3.26-.149-4.771-1.699-4.919-4.92-.058-1.265-.07-1.644-.07-4.849 0-3.204.013-3.583.07-4.849.149-3.227 1.664-4.771 4.919-4.919 1.266-.057 1.645-.069 4.849-.069zm0-2.163c-3.259 0-3.667.014-4.947.072-4.358.2-6.78 2.618-6.98 6.98-.059 1.281-.073 1.689-.073 4.948 0 3.259.014 3.668.072 4.948.2 4.358 2.618 6.78 6.98 6.98 1.281.058 1.689.072 4.948.072 3.259 0 3.668-.014 4.948-.072 4.354-.2 6.782-2.618 6.979-6.98.059-1.28.073-1.689.073-4.948 0-3.259-.014-3.667-.072-4.947-.196-4.354-2.617-6.78-6.979-6.98-1.281-.059-1.69-.073-4.949-.073zm0 5.838c-3.403 0-6.162 2.759-6.162 6.162s2.759 6.163 6.162 6.163 6.162-2.759 6.162-6.163c0-3.403-2.759-6.162-6.162-6.162zm0 10.162c-2.209 0-4-1.79-4-4 0-2.209 1.791-4 4-4s4 1.791 4 4c0 2.21-1.791 4-4 4zm6.406-11.845c-.796 0-1.441.645-1.441 1.44s.645 1.44 1.441 1.44c.795 0 1.439-.645 1.439-1.44s-.644-1.44-1.439-1.44z"/>
                    </svg>
                    @<?= htmlspecialchars($developer['instagram']) ?>
                </a>
            </div>
        </header>
        
        <form id="addTaskForm" class="add-form">
            <div class="form-group grow">
                <label>Görev</label>
                <input type="text" name="title" placeholder="Yeni görev..." required>
            </div>
            <div class="form-group grow">
                <label>Açıklama</label>
                <input type="text" name="description" placeholder="Opsiyonel">
            </div>
            <div class="form-group">
                <label>Öncelik</label>
                <select name="priority">
                    <option value="low">Düşük</option>
                    <option value="medium" selected>Orta</option>
                    <option value="high">Yüksek</option>
                </select>
            </div>
            <div class="form-group">
                <label>Renk</label>
                <input type="color" name="color" value="#3b82f6" class="color-input">
            </div>
            <button type="submit" class="btn">Ekle</button>
        </form>
        
        <div class="board">
            <?php
            $columns = [
                'todo' => ['title' => 'Yapılacak', 'class' => 'column-todo'],
                'inprogress' => ['title' => 'Devam Ediyor', 'class' => 'column-inprogress'],
                'done' => ['title' => 'Tamamlandı', 'class' => 'column-done']
            ];
            
            foreach ($columns as $status => $col):
            ?>
            <div class="column <?= $col['class'] ?>">
                <div class="column-header">
                    <div class="column-title">
                        <span class="column-dot"></span>
                        <h2><?= $col['title'] ?></h2>
                    </div>
                    <span class="column-count"><?= count($groupedTasks[$status]) ?></span>
                </div>
                <div class="task-list" data-status="<?= $status ?>">
                    <?php if (empty($groupedTasks[$status])): ?>
                        <div class="empty">Görev yok</div>
                    <?php else: ?>
                        <?php foreach ($groupedTasks[$status] as $t): ?>
                            <div class="task-card" draggable="true" data-task-id="<?= $t['id'] ?>" style="--task-color: <?= htmlspecialchars($t['color']) ?>">
                                <div class="task-top">
                                    <span class="task-title"><?= htmlspecialchars($t['title']) ?></span>
                                    <span class="task-priority priority-<?= $t['priority'] ?>"><?= $priorityLabels[$t['priority']] ?></span>
                                </div>
                                <?php if ($t['description']): ?>
                                    <p class="task-desc"><?= htmlspecialchars($t['description']) ?></p>
                                <?php endif; ?>
                                <div class="task-footer">
                                    <span class="task-date"><?= date('j M', strtotime($t['created_at'])) ?></span>
                                    <div class="task-actions">
                                        <button class="task-edit" title="Düzenle">
                                            <svg width="14" height="14" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                                <path d="M11 4H4a2 2 0 00-2 2v14a2 2 0 002 2h14a2 2 0 002-2v-7"/>
                                                <path d="M18.5 2.5a2.121 2.121 0 013 3L12 15l-4 1 1-4 9.5-9.5z"/>
                                            </svg>
                                        </button>
                                        <button class="task-delete" title="Sil">
                                            <svg width="14" height="14" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                                <path d="M3 6h18M8 6V4a2 2 0 012-2h4a2 2 0 012 2v2m3 0v14a2 2 0 01-2 2H7a2 2 0 01-2-2V6h14z"/>
                                            </svg>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
    
    <footer>
        <div class="footer-content">
            <p>
                Developed by <a href="<?= htmlspecialchars($developer['website']) ?>" target="_blank"><?= htmlspecialchars($developer['name']) ?></a>
                ·
                <a href="https://instagram.com/<?= htmlspecialchars($developer['instagram']) ?>" target="_blank">
                    <svg viewBox="0 0 24 24" fill="currentColor">
                        <path d="M12 2.163c3.204 0 3.584.012 4.85.07 3.252.148 4.771 1.691 4.919 4.919.058 1.265.069 1.645.069 4.849 0 3.205-.012 3.584-.069 4.849-.149 3.225-1.664 4.771-4.919 4.919-1.266.058-1.644.07-4.85.07-3.204 0-3.584-.012-4.849-.07-3.26-.149-4.771-1.699-4.919-4.92-.058-1.265-.07-1.644-.07-4.849 0-3.204.013-3.583.07-4.849.149-3.227 1.664-4.771 4.919-4.919 1.266-.057 1.645-.069 4.849-.069zm0-2.163c-3.259 0-3.667.014-4.947.072-4.358.2-6.78 2.618-6.98 6.98-.059 1.281-.073 1.689-.073 4.948 0 3.259.014 3.668.072 4.948.2 4.358 2.618 6.78 6.98 6.98 1.281.058 1.689.072 4.948.072 3.259 0 3.668-.014 4.948-.072 4.354-.2 6.782-2.618 6.979-6.98.059-1.28.073-1.689.073-4.948 0-3.259-.014-3.667-.072-4.947-.196-4.354-2.617-6.78-6.979-6.98-1.281-.059-1.69-.073-4.949-.073zm0 5.838c-3.403 0-6.162 2.759-6.162 6.162s2.759 6.163 6.162 6.163 6.162-2.759 6.162-6.163c0-3.403-2.759-6.162-6.162-6.162zm0 10.162c-2.209 0-4-1.79-4-4 0-2.209 1.791-4 4-4s4 1.791 4 4c0 2.21-1.791 4-4 4zm6.406-11.845c-.796 0-1.441.645-1.441 1.44s.645 1.44 1.441 1.44c.795 0 1.439-.645 1.439-1.44s-.644-1.44-1.439-1.44z"/>
                    </svg>
                    @<?= htmlspecialchars($developer['instagram']) ?>
                </a>
            </p>
        </div>
    </footer>
    
    <script>
    const BASE_URL = '';
    let draggedCard = null;
    let placeholder = null;

    document.addEventListener('DOMContentLoaded', function() {
        initDragAndDrop();
        initFormSubmit();
        initTaskActions();
        updateTaskCounts();
    });

    function initDragAndDrop() {
        document.querySelectorAll('.task-card').forEach(card => {
            card.addEventListener('dragstart', handleDragStart);
            card.addEventListener('dragend', handleDragEnd);
        });
        
        document.querySelectorAll('.task-list').forEach(list => {
            list.addEventListener('dragover', handleDragOver);
            list.addEventListener('drop', handleDrop);
            list.addEventListener('dragenter', handleDragEnter);
            list.addEventListener('dragleave', handleDragLeave);
        });
    }

    function handleDragStart(e) {
        draggedCard = this;
        
        // Placeholder oluştur
        placeholder = document.createElement('div');
        placeholder.className = 'task-placeholder';
        placeholder.style.height = this.offsetHeight + 'px';
        
        setTimeout(() => {
            this.classList.add('dragging');
            this.parentNode.insertBefore(placeholder, this.nextSibling);
        }, 0);
        
        e.dataTransfer.effectAllowed = 'move';
    }

    function handleDragEnd() {
        this.classList.remove('dragging');
        
        // Placeholder kaldır
        if (placeholder && placeholder.parentNode) {
            placeholder.parentNode.removeChild(placeholder);
        }
        placeholder = null;
        
        document.querySelectorAll('.column').forEach(c => c.classList.remove('drag-over'));
        draggedCard = null;
    }

    function handleDragEnter(e) {
        e.preventDefault();
        this.closest('.column').classList.add('drag-over');
    }

    function handleDragLeave(e) {
        if (!this.contains(e.relatedTarget)) {
            this.closest('.column').classList.remove('drag-over');
        }
    }

    function handleDragOver(e) {
        e.preventDefault();
        if (!draggedCard || !placeholder) return;
        
        const afterElement = getDragAfterElement(this, e.clientY);
        
        if (afterElement) {
            this.insertBefore(placeholder, afterElement);
        } else {
            this.appendChild(placeholder);
        }
    }

    function handleDrop(e) {
        e.preventDefault();
        this.closest('.column').classList.remove('drag-over');
        if (!draggedCard) return;
        
        const taskId = draggedCard.dataset.taskId;
        const newStatus = this.dataset.status;
        
        // Placeholder yerine kartı koy
        if (placeholder && placeholder.parentNode) {
            placeholder.parentNode.insertBefore(draggedCard, placeholder);
            placeholder.parentNode.removeChild(placeholder);
        }
        
        // Animasyon ekle
        draggedCard.classList.remove('dragging');
        draggedCard.classList.add('just-dropped');
        
        // Tamamlandı kolonuna bırakıldıysa konfeti
        if (newStatus === 'done') {
            createConfetti(e.clientX, e.clientY);
        }
        
        setTimeout(() => {
            draggedCard.classList.remove('just-dropped');
        }, 600);
        
        const cards = [...this.querySelectorAll('.task-card')];
        const newPosition = cards.indexOf(draggedCard);
        
        fetch(BASE_URL + 'actions/update_status.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ task_id: taskId, status: newStatus, position: newPosition })
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                showToast(newStatus === 'done' ? '🎉 Görev tamamlandı!' : 'Görev taşındı', 'success');
                updateTaskCounts();
            }
        })
        .catch(() => showToast('Bağlantı hatası', true));
        
        placeholder = null;
    }

    function getDragAfterElement(container, y) {
        const elements = [...container.querySelectorAll('.task-card:not(.dragging)')];
        return elements.reduce((closest, child) => {
            const box = child.getBoundingClientRect();
            const offset = y - box.top - box.height / 2;
            if (offset < 0 && offset > closest.offset) return { offset, element: child };
            return closest;
        }, { offset: -Infinity }).element;
    }

    // KONFETİ EFEKT
    function createConfetti(x, y) {
        const colors = ['#f59e0b', '#22c55e', '#3b82f6', '#ef4444', '#8b5cf6', '#ec4899'];
        
        for (let i = 0; i < 20; i++) {
            const confetti = document.createElement('div');
            confetti.className = 'confetti';
            confetti.style.left = (x + (Math.random() - 0.5) * 100) + 'px';
            confetti.style.top = (y + (Math.random() - 0.5) * 50) + 'px';
            confetti.style.background = colors[Math.floor(Math.random() * colors.length)];
            confetti.style.borderRadius = Math.random() > 0.5 ? '50%' : '0';
            confetti.style.transform = `rotate(${Math.random() * 360}deg)`;
            document.body.appendChild(confetti);
            
            setTimeout(() => confetti.remove(), 1000);
        }
    }

    function initFormSubmit() {
        const form = document.getElementById('addTaskForm');
        if (!form) return;
        
        form.addEventListener('submit', async (e) => {
            e.preventDefault();
            const btn = form.querySelector('button[type="submit"]');
            btn.disabled = true;
            btn.textContent = '...';
            
            const data = {
                title: form.title.value.trim(),
                description: form.description.value.trim(),
                priority: form.priority.value,
                color: form.color.value
            };
            
            if (!data.title) {
                showToast('Görev başlığı gerekli', true);
                btn.disabled = false;
                btn.textContent = 'Ekle';
                return;
            }
            
            try {
                const res = await fetch(BASE_URL + 'actions/add_task.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(data)
                });
                const result = await res.json();
                
                if (result.success) {
                    showToast('Görev eklendi', 'success');
                    form.reset();
                    form.color.value = '#3b82f6';
                    addTaskToBoard(result.task);
                    updateTaskCounts();
                } else {
                    showToast(result.message || 'Hata oluştu', true);
                }
            } catch (err) {
                showToast('Bağlantı hatası', true);
            }
            
            btn.disabled = false;
            btn.textContent = 'Ekle';
        });
    }

    function addTaskToBoard(task) {
        const list = document.querySelector(`.task-list[data-status="${task.status}"]`);
        if (!list) return;
        
        const empty = list.querySelector('.empty');
        if (empty) empty.remove();
        
        const priorityLabels = { low: 'Düşük', medium: 'Orta', high: 'Yüksek' };
        const date = new Date(task.created_at);
        const dateStr = date.toLocaleDateString('tr-TR', { day: 'numeric', month: 'short' });
        
        const card = document.createElement('div');
        card.className = 'task-card just-added';
        card.draggable = true;
        card.dataset.taskId = task.id;
        card.style.setProperty('--task-color', task.color);
        
        card.innerHTML = `
            <div class="task-top">
                <span class="task-title">${escapeHtml(task.title)}</span>
                <span class="task-priority priority-${task.priority}">${priorityLabels[task.priority]}</span>
            </div>
            ${task.description ? `<p class="task-desc">${escapeHtml(task.description)}</p>` : ''}
            <div class="task-footer">
                <span class="task-date">${dateStr}</span>
                <div class="task-actions">
                    <button class="task-edit" title="Düzenle"><svg width="14" height="14" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24"><path d="M11 4H4a2 2 0 00-2 2v14a2 2 0 002 2h14a2 2 0 002-2v-7"/><path d="M18.5 2.5a2.121 2.121 0 013 3L12 15l-4 1 1-4 9.5-9.5z"/></svg></button>
                    <button class="task-delete" title="Sil"><svg width="14" height="14" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24"><path d="M3 6h18M8 6V4a2 2 0 012-2h4a2 2 0 012 2v2m3 0v14a2 2 0 01-2 2H7a2 2 0 01-2-2V6h14z"/></svg></button>
                </div>
            </div>
        `;
        
        setTimeout(() => card.classList.remove('just-added'), 500);
        
        card.addEventListener('dragstart', handleDragStart);
        card.addEventListener('dragend', handleDragEnd);
        list.insertBefore(card, list.firstChild);
    }

    function initTaskActions() {
        document.addEventListener('click', (e) => {
            const deleteBtn = e.target.closest('.task-delete');
            const editBtn = e.target.closest('.task-edit');
            
            if (deleteBtn) {
                const card = deleteBtn.closest('.task-card');
                if (card && confirm('Bu görevi silmek istediğinize emin misiniz?')) deleteTask(card);
            }
            
            if (editBtn) {
                const card = editBtn.closest('.task-card');
                if (card) openEditModal(card);
            }
        });
    }

    async function deleteTask(card) {
        const taskId = card.dataset.taskId;
        
        try {
            const res = await fetch(BASE_URL + 'actions/delete_task.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ task_id: taskId })
            });
            const data = await res.json();
            
            if (data.success) {
                card.classList.add('removing');
                setTimeout(() => {
                    card.remove();
                    updateTaskCounts();
                    checkEmptyState();
                }, 300);
                showToast('Görev silindi', 'success');
            } else {
                showToast(data.message || 'Silinemedi', true);
            }
        } catch (err) {
            showToast('Bağlantı hatası', true);
        }
    }

    function openEditModal(card) {
        const taskId = card.dataset.taskId;
        const title = card.querySelector('.task-title').textContent;
        const descEl = card.querySelector('.task-desc');
        const desc = descEl ? descEl.textContent : '';
        const priorityClass = card.querySelector('.task-priority').className;
        const priority = priorityClass.includes('low') ? 'low' : priorityClass.includes('high') ? 'high' : 'medium';
        const color = getComputedStyle(card).getPropertyValue('--task-color').trim() || '#3b82f6';
        
        const modal = document.createElement('div');
        modal.className = 'modal-overlay';
        modal.innerHTML = `
            <div class="modal">
                <div class="modal-header">
                    <h3>Görevi Düzenle</h3>
                    <button class="modal-close">&times;</button>
                </div>
                <form class="modal-form">
                    <input type="hidden" name="task_id" value="${taskId}">
                    <div class="modal-field">
                        <label>Başlık</label>
                        <input type="text" name="title" value="${escapeHtml(title)}" required>
                    </div>
                    <div class="modal-field">
                        <label>Açıklama</label>
                        <input type="text" name="description" value="${escapeHtml(desc)}">
                    </div>
                    <div class="modal-row">
                        <div class="modal-field">
                            <label>Öncelik</label>
                            <select name="priority">
                                <option value="low" ${priority === 'low' ? 'selected' : ''}>Düşük</option>
                                <option value="medium" ${priority === 'medium' ? 'selected' : ''}>Orta</option>
                                <option value="high" ${priority === 'high' ? 'selected' : ''}>Yüksek</option>
                            </select>
                        </div>
                        <div class="modal-field">
                            <label>Renk</label>
                            <input type="color" name="color" value="${color}" class="color-input">
                        </div>
                    </div>
                    <div class="modal-actions">
                        <button type="button" class="btn-cancel">İptal</button>
                        <button type="submit" class="btn">Kaydet</button>
                    </div>
                </form>
            </div>
        `;
        
        document.body.appendChild(modal);
        
        modal.querySelector('.modal-close').onclick = () => modal.remove();
        modal.querySelector('.btn-cancel').onclick = () => modal.remove();
        modal.onclick = (e) => { if (e.target === modal) modal.remove(); };
        
        modal.querySelector('.modal-form').onsubmit = async (e) => {
            e.preventDefault();
            const form = e.target;
            
            const data = {
                task_id: form.task_id.value,
                title: form.title.value.trim(),
                description: form.description.value.trim(),
                priority: form.priority.value,
                color: form.color.value
            };
            
            try {
                const res = await fetch(BASE_URL + 'actions/edit_task.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(data)
                });
                const result = await res.json();
                
                if (result.success) {
                    updateCardUI(card, result.task);
                    modal.remove();
                    showToast('Görev güncellendi', 'success');
                } else {
                    showToast(result.message || 'Güncellenemedi', true);
                }
            } catch (err) {
                showToast('Bağlantı hatası', true);
            }
        };
    }

    function updateCardUI(card, task) {
        const priorityLabels = { low: 'Düşük', medium: 'Orta', high: 'Yüksek' };
        card.querySelector('.task-title').textContent = task.title;
        card.style.setProperty('--task-color', task.color);
        const priorityEl = card.querySelector('.task-priority');
        priorityEl.className = `task-priority priority-${task.priority}`;
        priorityEl.textContent = priorityLabels[task.priority];
        
        let descEl = card.querySelector('.task-desc');
        if (task.description) {
            if (descEl) descEl.textContent = task.description;
            else {
                const newDesc = document.createElement('p');
                newDesc.className = 'task-desc';
                newDesc.textContent = task.description;
                card.querySelector('.task-top').after(newDesc);
            }
        } else if (descEl) descEl.remove();
        
        // Güncelleme animasyonu
        card.style.animation = 'none';
        card.offsetHeight;
        card.style.animation = 'drop-bounce 0.4s ease';
    }

    function updateTaskCounts() {
        document.querySelectorAll('.column').forEach(col => {
            const count = col.querySelectorAll('.task-card').length;
            col.querySelector('.column-count').textContent = count;
        });
    }

    function checkEmptyState() {
        document.querySelectorAll('.task-list').forEach(list => {
            if (list.querySelectorAll('.task-card').length === 0 && !list.querySelector('.empty')) {
                const empty = document.createElement('div');
                empty.className = 'empty';
                empty.textContent = 'Görev yok';
                list.appendChild(empty);
            }
        });
    }

    function showToast(msg, type = '') {
        const existing = document.querySelector('.toast');
        if (existing) existing.remove();
        const toast = document.createElement('div');
        toast.className = 'toast' + (type === true ? ' error' : type ? ` ${type}` : '');
        toast.textContent = msg;
        document.body.appendChild(toast);
        setTimeout(() => toast.remove(), 2500);
    }

    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    </script>
</body>
</html>